<?php

namespace App\Http\Controllers;

use App\Suppliers;
use App\GoodReceipts;
use App\GoodReceiptTruck;
use App\GoodReceiptDetail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;

class GoodReceiptsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $good_receipts_truck = GoodReceiptTruck::with(['goodReceipts.goodReceiptDetails', 'goodReceipts.supplier'])->get();

        return view('goods_receipts.index', compact('good_receipts_truck'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $suppliers = Suppliers::all();
        return view('goods_receipts.create', compact('suppliers'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
    */

    public function store(Request $request)
    {
        // Validate the request data
        $request->validate([
            'goods_date' => 'required|date',
            'p_no' => 'nullable|string|max:255',
            'truck_number' => 'required|string|max:255',
            'suppliers' => 'required|exists:suppliers,id_supplier',
            'builty_number' => 'nullable|string|max:255',
            'rent_received' => 'nullable|numeric',
            'goods_status' => 'required|in:1,0',
            'details' => 'nullable|string',
            'items.*' => 'required|string|max:255',
            'marka.*' => 'nullable|string|max:255',
            'bori.*' => 'nullable|numeric',
            'weight.*' => 'nullable|numeric',
            'rate.*' => 'nullable|numeric',
            'remarks.*' => 'nullable|string',
        ]);

        // Create the GoodReceiptTruck entry
        $truck = GoodReceiptTruck::create(['truck_no' => $request->truck_number]);

        // Create the GoodReceipts entry
        $goodReceipt = GoodReceipts::create([
            'business_id' => auth()->user()->business_id,
            'supplier_id' => $request->suppliers,
            'good_receipts_truck_id' => $truck->id_truck,
            'date' => $request->goods_date,
            'p_no' => $request->p_no,
            'builty_no' => $request->builty_number,
            'rent_recieve' => $request->rent_received,
            'details' => $request->details,
            'status' => $request->goods_status,
            'created_by' => auth()->user()->id,
        ]);

        // Store GoodReceiptDetails
        foreach ($request->items as $key => $item) {
            GoodReceiptDetail::create([
                'good_receipts_id' => $goodReceipt->id_good_rec,
                'truck_id' => $truck->id_truck,
                'goods_name' => $item,
                'marka' => $request->marka[$key] ?? null,
                'bori' => $request->bori[$key] ?? null,
                'weight' => $request->weight[$key] ?? null,
                'rate' => $request->rate[$key] ?? null,
                'remarks' => $request->remarks[$key] ?? null,
            ]);
        }

        return redirect()->route('good_receipts.list')->with('success', 'Goods Receipt created successfully!');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\GoodReceipts  $goodReceipts
     * @return \Illuminate\Http\Response
     */
    public function show(GoodReceipts $goodReceipts)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\GoodReceipts  $goodReceipts
     * @return \Illuminate\Http\Response
    */
    public function edit($id)
    {
        $id = Crypt::decrypt($id);
        $goodReceipt = GoodReceipts::with('goodReceiptDetails', 'goodReceiptTruck')->findOrFail($id);
        
        $suppliers = Suppliers::all();
        
        return view('goods_receipts.edit', compact('goodReceipt', 'suppliers'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\GoodReceipts  $goodReceipts
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // Validate the request data
        $request->validate([
            'goods_date' => 'required|date',
            'p_no' => 'nullable|string|max:255',
            'truck_number' => 'required|string|max:255',
            'suppliers' => 'required|exists:suppliers,id_supplier',
            'builty_number' => 'nullable|string|max:255',
            'rent_received' => 'nullable|numeric',
            'goods_status' => 'required|in:1,0',
            'details' => 'nullable|string',
            'items.*' => 'required|string|max:255',
            'marka.*' => 'nullable|string|max:255',
            'bori.*' => 'nullable|numeric',
            'weight.*' => 'nullable|numeric',
            'rate.*' => 'nullable|numeric',
            'remarks.*' => 'nullable|string',
        ]);

        $goodReceipt = GoodReceipts::findOrFail($id);

        $truck = $goodReceipt->goodReceiptTruck;
        $truck->update([
            'truck_no' => $request->truck_number
        ]);

        $goodReceipt->update([
            'supplier_id' => $request->suppliers,
            'good_receipts_truck_id' => $truck->id_truck,
            'date' => $request->goods_date,
            'p_no' => $request->p_no,
            'builty_no' => $request->builty_number,
            'rent_recieve' => $request->rent_received,
            'details' => $request->details,
            'status' => $request->goods_status,
        ]);

        $goodReceipt->goodReceiptDetails()->delete();

        foreach ($request->items as $key => $item) {
            GoodReceiptDetail::create([
                'good_receipts_id' => $goodReceipt->id_good_rec,
                'truck_id' => $truck->id_truck,
                'goods_name' => $item,
                'marka' => $request->marka[$key] ?? null,
                'bori' => $request->bori[$key] ?? null,
                'weight' => $request->weight[$key] ?? null,
                'rate' => $request->rate[$key] ?? null,
                'remarks' => $request->remarks[$key] ?? null,
            ]);
        }

        return redirect()->route('good_receipts.list')->with('success', 'Goods Receipt updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\GoodReceipts  $goodReceipts
     * @return \Illuminate\Http\Response
    */
    public function destroy($id)
    {
        $goodReceiptId = Crypt::decrypt($id);

        $goodReceipt = GoodReceipts::findOrFail($goodReceiptId);

        $goodReceipt->goodReceiptDetails()->delete();

        $goodReceipt->goodReceiptTruck()->delete();

        $goodReceipt->delete();

        return redirect()->route('good_receipts.list')->with('success', 'Goods Receipt deleted successfully!');
    }

}
