<?php

namespace App\Http\Controllers;

use App\BusinessPartner;
use App\Customer;
use App\Employees;
use App\TransactionAccount;
use App\Voucher;
use App\VouchersUniqueNumber;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;

class VoucherController extends Controller
{
    private const UNITS = ['Zero', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine'];
    private const TEENS = ['Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen', 'Seventeen', 'Eighteen', 'Nineteen'];
    private const TENS = ['Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];
    private const THOUSANDS = ['Thousand', 'Million', 'Billion', 'Trillion'];


    public function index()
    {
        $vouchersNumber = VouchersUniqueNumber::with('vouchers')->where('voucher_type', 'Journal Voucher')->get();
        return view('voucher.index', compact('vouchersNumber'));
    }

    public function create()
    {
        // Fetch employees data
        $employees = Employees::all();
        $accounts = TransactionAccount::all();
        return view('voucher.create', compact('employees', 'accounts'));
    }


    public function store(Request $request)
    {
        // Validate the incoming data
        $request->validate([
            'voucherDate.*' => 'required|date',
            'accountNo.*' => 'required|string',
            'accountName.*' => 'required|string',
            'paymentMode.*' => 'required|string',
            'instrumentAmount.*' => 'nullable|string',
            'netAmount.*' => 'required|numeric',
            'noteRemarks.*' => 'nullable|string',
            'debitAMount.*' => 'nullable|numeric',
            'creditAmount.*' => 'nullable|numeric',
            'businessPartnerName.*' => 'nullable|string',
            'taxDebitYesNo.*' => 'required|string',
            'transactionAccountId.*' => 'required|string',
            'transactionAccountName.*' => 'required|string',
            'vDescription.*' => 'nullable|string',
            'voucherStatus.*' => 'required|string',
        ]);

        // Generate a unique journal voucher number (e.g., JV-XXXXX)
        $uniqueVoucherNumber = 'JV-' . str_pad(rand(0, 99999), 5, '0', STR_PAD_LEFT);

        // Create a new VouchersUniqueNumber record
        $voucherUniqueNumber = new VouchersUniqueNumber();
        $voucherUniqueNumber->voucher_type = 'Journal Voucher';
        $voucherUniqueNumber->voucher_number = $uniqueVoucherNumber;
        $voucherUniqueNumber->save();  // Save the unique voucher number record first

        // Loop through each row and create a voucher record
        for ($i = 0; $i < count($request->accountNo); $i++) {
            $voucher = new Voucher();
            $voucher->vouchers_unique_number_id = $voucherUniqueNumber->id_vouchers_unique_number;  // Associate with unique voucher number
            $voucher->date = $request->voucherDate[$i];
            $voucher->account_no = $request->accountNo[$i];
            $voucher->account_title = $request->accountName[$i];
            $voucher->payment_mode = $request->paymentMode[$i];
            $voucher->instrument = $request->instrumentAmount[$i];
            $voucher->amount = $request->netAmount[$i];
            $voucher->remarks = $request->noteRemarks[$i];
            $voucher->debit = $request->debitAMount[$i];
            $voucher->credit = $request->creditAmount[$i];
            $voucher->partner_name = $request->businessPartnerName[$i];
            $voucher->tax_debit = $request->taxDebitYesNo[$i];
            $voucher->transaction_acc_no = $request->transactionAccountId[$i];
            $voucher->transaction_acc_name = $request->transactionAccountName[$i];
            $voucher->desciption = $request->vDescription[$i];
            $voucher->status = $request->voucherStatus[$i];
            // Save the voucher
            $voucher->save();
        }

        // Redirect back with a success message
        return redirect()->route('voucher.list')->with('success', 'Vouchers have been created successfully.');
    }


    public function view($id)
    {
        $id = Crypt::decrypt($id);
        $vouchersNumber = VouchersUniqueNumber::with('vouchers')->where('voucher_type', 'Journal Voucher')->where('id_vouchers_unique_number', $id)->first();
        $voucherSingle = $vouchersNumber->vouchers->first();
        return view('voucher.view', compact('vouchersNumber', 'voucherSingle'));
    }

    public function edit($encryptedId)
    {
        $id = Crypt::decrypt($encryptedId);
        $vouchersNumber = VouchersUniqueNumber::with('vouchers')->where('voucher_type', 'Journal Voucher')->findOrFail($id);
        $voucherSingle = $vouchersNumber->vouchers->first();

        // Fetch employees and accounts for the form
        $employees = Employees::all();
        $accounts = TransactionAccount::all();

        return view('voucher.edit', compact('employees', 'accounts', 'vouchersNumber', 'voucherSingle'));
    }

    public function update(Request $request, $encryptedId)
    {
        // Validate the request data
        $request->validate([
            'voucherDate.*' => 'required|date',
            'accountNo.*' => 'required|string',
            'accountName.*' => 'required|string',
            'paymentMode.*' => 'required|string',
            'instrumentAmount.*' => 'nullable|string',
            'netAmount.*' => 'required|numeric',
            'noteRemarks.*' => 'nullable|string',
            'debitAMount.*' => 'nullable|numeric',
            'creditAmount.*' => 'nullable|numeric',
            'businessPartnerName.*' => 'nullable|string',
            'taxDebitYesNo.*' => 'required|string',
            'transactionAccountId.*' => 'required|string',
            'transactionAccountName.*' => 'required|string',
            'vDescription.*' => 'nullable|string',
            'voucherStatus.*' => 'required|string',
            'voucherIds.*' => 'nullable|integer'
        ]);


        if ($request->has('deletedVouchers')) {
            $deletedVouchers = explode(',', $request->input('deletedVouchers'));
            Voucher::whereIn('id_vouchers', $deletedVouchers)->delete();
        }

        $id = Crypt::decrypt($encryptedId);
        $voucherUniqueNumber = VouchersUniqueNumber::with('vouchers')->findOrFail($id);
        $existingVouchers = Voucher::where('vouchers_unique_number_id', $voucherUniqueNumber->id_vouchers_unique_number)->get();

        // Track new vouchers to be inserted
        $newVouchersData = [];

        // Track IDs to update and compare against existing ones
        $voucherIdsToUpdate = $request->input('voucherIds', []);
        $existingVoucherIds = $existingVouchers->pluck('id_vouchers')->toArray();

        // Process the vouchers from the request
        foreach ($request->input('voucherDate', []) as $index => $voucherDate) {
            $voucherId = $voucherIdsToUpdate[$index] ?? null;

            // Add new voucher
            $newVouchersData[] = [
                'vouchers_unique_number_id' => $voucherUniqueNumber->id_vouchers_unique_number,
                'date' => $voucherDate,
                'account_no' => $request->input('accountNo.' . $index),
                'account_title' => $request->input('accountName.' . $index),
                'payment_mode' => $request->input('paymentMode.' . $index),
                'instrument' => $request->input('instrumentAmount.' . $index),
                'amount' => $request->input('netAmount.' . $index),
                'remarks' => $request->input('noteRemarks.' . $index),
                'debit' => $request->input('debitAMount.' . $index),
                'credit' => $request->input('creditAmount.' . $index),
                'partner_name' => $request->input('businessPartnerName.' . $index),
                'tax_debit' => $request->input('taxDebitYesNo.' . $index),
                'transaction_acc_no' => $request->input('transactionAccountId.' . $index),
                'transaction_acc_name' => $request->input('transactionAccountName.' . $index),
                'desciption' => $request->input('vDescription.' . $index),
                'status' => $request->input('voucherStatus.' . $index)
            ];
            // dd($voucherId, $existingVoucherIds , $newVouchersData);
        }

        // Insert new vouchers
        if (!empty($newVouchersData)) {
            Voucher::insert($newVouchersData);
        }

        return redirect()->route('voucher.list')->with('success', 'Vouchers have been updated, and new vouchers have been added successfully.');
    }

    // public function print($id)
    // {
    //     $id = Crypt::decrypt($id);
    //     $vouchersNumber = VouchersUniqueNumber::with('vouchers')->where('voucher_type', 'Journal Voucher')->where('id_vouchers_unique_number', $id)->first();
    //     $voucherSingle = $vouchersNumber->vouchers->first();
    //     return view('voucher.print', compact('vouchersNumber', 'voucherSingle'));
    // }

    // In VoucherController.php
    public function print($id)
    {
        $id = Crypt::decrypt($id);
        $voucherNumber = VouchersUniqueNumber::with('vouchers')->where('voucher_type', 'Journal Voucher')->where('id_vouchers_unique_number', $id)->first();
        // $voucherNumber = VouchersUniqueNumber::with('vouchers')->findOrFail(Crypt::decrypt($id));
        
        return view('voucher.print', compact('voucherNumber'));
    }

    public function getPartnerData(Request $request)
    {
        $type = $request->get('type');

        if ($type === 'employee') {
            $data = Employees::pluck('employee_name', 'id_employee'); // Adjust fields as necessary
        } elseif ($type === 'customer') {
            $data = Customer::pluck('customer_name', 'id_customers'); // Adjust fields as necessary
        } else {
            $data = []; // Handle other types later if necessary
        }

        return response()->json($data);
    }

    public function saveVendor(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
        ]);

        // Save the new vendor to the business_partner table
        $vendor = new BusinessPartner();
        $vendor->name = $request->name;
        $vendor->save();

        // Return a JSON response with success and the vendor ID
        return response()->json([
            'success' => true,
            'vendor_id' => $vendor->id_business_partner
        ]);
    }

    public function getTransactionAccounts(Request $request)
    {
        $type = $request->get('type');

        // Validate the type to ensure it's either 'cash' or 'bank'
        if (!in_array($type, ['cash', 'bank'])) {
            return response()->json([]);
        }

        // Fetch accounts based on the account type
        $accounts = TransactionAccount::where('account_type', $type)->pluck('transaction_account_name', 'id_transaction_account');

        return response()->json($accounts);
    }

    private function convertNumber($number)
    {
        if ($number == 0) {
            return 'Zero';
        }

        $number = (int)$number;
        $result = '';

        if ($number < 10) {
            $result = self::UNITS[$number];
        } elseif ($number < 20) {
            $result = self::TEENS[$number - 11];
        } elseif ($number < 100) {
            $result = self::TENS[intdiv($number, 10) - 2];
            if (($number % 10) > 0) {
                $result .= '-' . self::UNITS[$number % 10];
            }
        } elseif ($number < 1000) {
            $result = self::UNITS[intdiv($number, 100)] . ' Hundred';
            if (($number % 100) > 0) {
                $result .= ' and ' . $this->convertNumber($number % 100);
            }
        } elseif ($number < 1000000) {
            $result = $this->convertNumber(intdiv($number, 1000)) . ' Thousand';
            if (($number % 1000) > 0) {
                $result .= ' ' . $this->convertNumber($number % 1000);
            }
        } else {
            // Handle millions, billions, etc.
            foreach (self::THOUSANDS as $index => $word) {
                $divider = pow(1000, $index + 1);
                if ($number >= $divider) {
                    $result = $this->convertNumber(intdiv($number, $divider)) . ' ' . $word;
                    if (($number % $divider) > 0) {
                        $result .= ' ' . $this->convertNumber($number % $divider);
                    }
                    break;
                }
            }
        }

        return $result;
    }
}
